<?php
// Mostrar errores durante desarrollo
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);
$error = '';

// Función para validar MAC (tomada del segundo código)
function validarMAC($mac) {
    return preg_match('/^([0-9A-Fa-f]{2}[:-]){2,7}[0-9A-Fa-f]{2}$/', $mac);
}

// Función para validar WhatsApp (tomada del segundo código)
function validarWhatsApp($numero) {
    return preg_match('/^\+?[1-9][0-9]{7,14}$/', $numero);
}

// Función para generar código (mejorada)
function generarCodigo() {
    $bytes = random_bytes(6);
    return strtoupper(bin2hex($bytes));
}

// Conexiones a bases de datos
try {
    $mainDB = new SQLite3(__DIR__.'/.db.db');
    $codesDB = new SQLite3(__DIR__.'/codes_rgvip.db');
    $mainDB->busyTimeout(5000);
    $codesDB->busyTimeout(5000);
    $mainDB->exec("PRAGMA journal_mode=WAL");
    $codesDB->exec("PRAGMA journal_mode=WAL");
    
    // Crear tabla si no existe (tomado del segundo código)
    $codesDB->exec('CREATE TABLE IF NOT EXISTS activation_codes (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        mac_address TEXT NOT NULL CHECK(length(mac_address) > 0),
        username TEXT NOT NULL CHECK(length(username) > 0),
        password TEXT NOT NULL CHECK(length(password) > 0),
        dns_id INTEGER NOT NULL,
        dns_url TEXT NOT NULL,
        cliente_nombre TEXT NOT NULL CHECK(length(cliente_nombre) > 0),
        cliente_whatsapp TEXT NOT NULL CHECK(length(cliente_whatsapp) > 0),
        expiracion TEXT NOT NULL CHECK(date(expiracion) IS NOT NULL),
        code TEXT UNIQUE NOT NULL CHECK(length(code) = 12),
        is_protected INTEGER DEFAULT 0 CHECK(is_protected IN (0,1)),
        created_at TEXT DEFAULT CURRENT_TIMESTAMP
    )');
} catch(Exception $e) {
    header("Location: ../generar_publico.php?error=" . urlencode("Error conectando BD: " . $e->getMessage()));
    exit;
}

// Cargar lista de DNS (petición AJAX desde el formulario)
if ($_SERVER['REQUEST_METHOD'] === 'GET' && isset($_GET['ajax']) && $_GET['ajax'] === 'dns') {
    $lista = [];
    try {
        $res = $mainDB->query("SELECT id, title FROM dns ORDER BY title");
        while ($row = $res->fetchArray(SQLITE3_ASSOC)) {
            $lista[] = $row;
        }
        header('Content-Type: application/json');
        echo json_encode(['success' => true, 'dns' => $lista]);
    } catch(Exception $e) {
        header('Content-Type: application/json');
        echo json_encode(['success' => false, 'error' => $e->getMessage()]);
    }
    exit;
}

// Procesar envío del formulario
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    try {
        // Recoger datos
        $mac = strtoupper(trim($_POST['mac_address'] ?? ''));
        $user = trim($_POST['username'] ?? '');
        $pass = trim($_POST['password'] ?? '');
        $dns_id = intval($_POST['dns_id'] ?? 0);
        $cliente_nombre = trim($_POST['cliente_nombre'] ?? '');
        $cliente_whatsapp = trim($_POST['cliente_whatsapp'] ?? '');
        $expiracion = trim($_POST['expiracion'] ?? '');
        $is_protected = isset($_POST['is_protected']) ? 1 : 0;

        // Asegurar que WhatsApp tenga +
        if (strpos($cliente_whatsapp, '+') !== 0) {
            $cliente_whatsapp = '+' . $cliente_whatsapp;
        }

        // Validaciones mejoradas
        if (!validarMAC($mac)) {
            throw new Exception('MAC inválida (debe tener entre 3 y 7 pares separados por :)');
        }
        
        if (strlen($user) < 3 || strlen($user) > 20) {
            throw new Exception('Usuario: 3‑20 caracteres');
        }
        
        if (strlen($pass) < 4) {
            throw new Exception('Contraseña: mínimo 4 caracteres');
        }
        
        if ($dns_id <= 0) {
            throw new Exception('Seleccione un DNS');
        }
        
        if (strlen($cliente_nombre) < 2 || strlen($cliente_nombre) > 50) {
            throw new Exception('Nombre del cliente: 2‑50 caracteres');
        }
        
        if (!validarWhatsApp($cliente_whatsapp)) {
            throw new Exception('WhatsApp inválido');
        }
        
        if (!preg_match('/^\d{4}-\d{2}-\d{2}$/', $expiracion) || strtotime($expiracion) < time()) {
            throw new Exception('Fecha de expiración inválida o pasada');
        }

        // Formatear MAC (solo hex y ":")
        $mac = preg_replace('/[^0-9A-F]/i', '', $mac);
        $macFormateada = implode(':', str_split($mac, 2));

        // Iniciar transacción
        $codesDB->exec('BEGIN TRANSACTION');

        // Obtener URL del DNS
        $stmt = $mainDB->prepare("SELECT url FROM dns WHERE id = :id");
        $stmt->bindValue(':id', $dns_id, SQLITE3_INTEGER);
        $dnsData = $stmt->execute()->fetchArray(SQLITE3_ASSOC);

        if (!$dnsData) {
            throw new Exception('DNS no encontrado');
        }

        $dns_url = $dnsData['url'];
        $codigoGenerado = generarCodigo();

        // Insertar en codes_rgvip.db
        $sql = "INSERT INTO activation_codes
                (mac_address, username, password, dns_id, dns_url, cliente_nombre, cliente_whatsapp, expiracion, code, is_protected)
                VALUES
                (:mac, :user, :pass, :dns_id, :dns_url, :cliente_nombre, :cliente_whatsapp, :expiracion, :code, :is_protected)";
        $stmt2 = $codesDB->prepare($sql);
        $stmt2->bindValue(':mac', $macFormateada, SQLITE3_TEXT);
        $stmt2->bindValue(':user', $user, SQLITE3_TEXT);
        $stmt2->bindValue(':pass', $pass, SQLITE3_TEXT);
        $stmt2->bindValue(':dns_id', $dns_id, SQLITE3_INTEGER);
        $stmt2->bindValue(':dns_url', $dns_url, SQLITE3_TEXT);
        $stmt2->bindValue(':cliente_nombre', $cliente_nombre, SQLITE3_TEXT);
        $stmt2->bindValue(':cliente_whatsapp', $cliente_whatsapp, SQLITE3_TEXT);
        $stmt2->bindValue(':expiracion', $expiracion, SQLITE3_TEXT);
        $stmt2->bindValue(':code', $codigoGenerado, SQLITE3_TEXT);
        $stmt2->bindValue(':is_protected', $is_protected, SQLITE3_INTEGER);

        if (!$stmt2->execute()) {
            throw new Exception('Error al guardar el código: ' . $codesDB->lastErrorMsg());
        }

        $codesDB->exec('COMMIT');

        // Redirigir con datos para mostrar en frontend
        $params = [
            'success' => 'Código generado correctamente',
            'codigo' => $codigoGenerado,
            'mac' => $macFormateada,
            'whatsapp' => $cliente_whatsapp,
            'nombre' => $cliente_nombre,
            'expiracion' => $expiracion
        ];
        header('Location: ../generar_publico.php?' . http_build_query($params));
        exit;

    } catch(Exception $e) {
        $codesDB->exec('ROLLBACK');
        header('Location: ../generar_publico.php?error=' . urlencode($e->getMessage()));
        exit;
    }
}

// Cierre explícito
$mainDB->close();
$codesDB->close();